<?php
if (!defined('__TYPECHO_ROOT_DIR__')) exit;
/**
 * 文章图片加速插件
 * 
 * 功能：自定义图片加速服务，支持灵活配置
 * 
 * @package ImageAccelerator
 * @author 黑海洋
 * @version 2.3
 * @link https://www.upx8.com/ImageAccelerator
 */
class ImageAccelerator_Plugin implements Typecho_Plugin_Interface
{
    /**
     * 激活插件
     */
    public static function activate()
    {
        Typecho_Plugin::factory('Widget_Abstract_Contents')->contentEx = array('ImageAccelerator_Plugin', 'accelerateImages');
    }

    /**
     * 禁用插件
     */
    public static function deactivate()
    {
        // 清理可能的残留配置
    }

    /**
     * 插件配置
     */
    public static function config(Typecho_Widget_Helper_Form $form)
    {
        // 启用开关
        $enableAccelerator = new Typecho_Widget_Helper_Form_Element_Checkbox('enable', 
            array('enable' => '启用图片加速'), 
            array('enable'), 
            '是否开启图片加速服务'
        );
        $form->addInput($enableAccelerator);

        // 加速服务链接
        $acceleratorUrl = new Typecho_Widget_Helper_Form_Element_Text(
            'accelerator_url', 
            null, 
            'https://image.baidu.com/search/down?url=', 
            '图片加速服务链接', 
            '请填写完整的图片加速服务URL，例如：https://image.baidu.com/search/down?url='
        );
        $form->addInput($acceleratorUrl);

        // 加速服务链接白名单
        $acceleratorUrlWhitelist = new Typecho_Widget_Helper_Form_Element_Textarea(
            'accelerator_url_whitelist', 
            null, 
            '', 
            '加速服务链接白名单', 
            '每行一个加速服务链接的域名或路径片段。符合白名单的链接将跳过加速处理。'
        );
        $form->addInput($acceleratorUrlWhitelist);

        // 排除的图片扩展名
        $excludeExtensions = new Typecho_Widget_Helper_Form_Element_Text(
            'exclude_extensions', 
            null, 
            'svg,gif', 
            '排除的图片扩展名', 
            '不进行加速的图片扩展名，用逗号分隔。'
        );
        $form->addInput($excludeExtensions);

        // 新增选项：是否移除 http:// 和 https://
        $removeProtocol = new Typecho_Widget_Helper_Form_Element_Checkbox(
            'remove_protocol', 
            array('remove' => '去掉原图片链接的 http:// 和 https://'), 
            array(), 
            '是否去掉原图片链接的协议'
        );
        $form->addInput($removeProtocol);

        $enableLogging = new Typecho_Widget_Helper_Form_Element_Checkbox(
            'logging', 
            array('enable' => '启用日志记录'), 
            array(), 
            '是否记录图片加速处理日志'
        );
        $form->addInput($enableLogging);
    }

    /**
     * 个人配置
     */
    public static function personalConfig(Typecho_Widget_Helper_Form $form)
    {
        // 预留个人配置
    }

    /**
     * 图片加速处理
     */
    public static function accelerateImages($content, $lastResult)
    {
        if (empty($content)) return $content;

        $options = Helper::options()->plugin('ImageAccelerator');
        
        if (!isset($options->enable) || !in_array('enable', $options->enable)) {
            return $content;
        }

        $acceleratorUrl = isset($options->accelerator_url) ? 
            trim($options->accelerator_url) : 
            '';

        if (empty($acceleratorUrl)) {
            return $content;
        }

        $acceleratorUrlWhitelist = isset($options->accelerator_url_whitelist) ? 
            array_filter(explode("\n", $options->accelerator_url_whitelist)) : 
            [];

        $excludeExtensions = isset($options->exclude_extensions) ? 
            explode(',', $options->exclude_extensions) : 
            ['svg', 'gif'];

        $removeProtocol = isset($options->remove_protocol) && in_array('remove', $options->remove_protocol);

        $pattern = '/<img\s+[^>]*src=["\']((?:https?:\/\/|\/\/)[^"\']+)(?=["\'])[^>]*>/i';

        $content = preg_replace_callback($pattern, function($matches) use (
            $acceleratorUrl, 
            $acceleratorUrlWhitelist, 
            $excludeExtensions,
            $removeProtocol,
            $options
        ) {
            $originalUrl = $matches[1];
            
            // 检查是否在加速服务链接白名单中
            foreach ($acceleratorUrlWhitelist as $whitelisted) {
                if (strpos($originalUrl, trim($whitelisted)) !== false) {
                    return $matches[0];
                }
            }

            if (strpos($originalUrl, $acceleratorUrl) !== false) {
                return $matches[0];
            }

            if (strpos($originalUrl, '//') === 0) {
                $originalUrl = 'https:' . $originalUrl;
            }

            $parsedUrl = parse_url($originalUrl);
            if (!$parsedUrl || !isset($parsedUrl['scheme']) || !isset($parsedUrl['host'])) {
                return $matches[0];
            }

            $extension = strtolower(pathinfo($parsedUrl['path'], PATHINFO_EXTENSION));
            if (in_array($extension, $excludeExtensions)) {
                return $matches[0];
            }

            // 生成用于加速的URL
            $urlToAccelerate = $originalUrl;

            if ($removeProtocol) {
                $urlToAccelerate = preg_replace('/^https?:\/\//', '', $urlToAccelerate);
            }

            $urlToAccelerate = ltrim($urlToAccelerate, '=');

            // 生成加速链接
            $acceleratedUrl = $acceleratorUrl . $urlToAccelerate;

            if (isset($options->logging) && in_array('enable', $options->logging)) {
                self::writeLog("加速图片: $originalUrl -> $acceleratedUrl");
            }

            return str_replace($originalUrl, $acceleratedUrl, $matches[0]);
        }, $content);

        return $content;
    }

    /**
     * 日志记录方法
     */
    private static function writeLog($message)
    {
        $logFile = __TYPECHO_ROOT_DIR__ . '/usr/plugins/ImageAccelerator/image_accelerator.log';
        
        $logDir = dirname($logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }

        file_put_contents($logFile, 
            date('[Y-m-d H:i:s] ') . $message . PHP_EOL, 
            FILE_APPEND
        );
    }
}